// DesktopOverlayHostWnd.cpp : implementation file
//
// created by Unwinder
/////////////////////////////////////////////////////////////////////////////
#include "stdafx.h"
#include "DesktopOverlayHostWnd.h"
#include "DesktopOverlayHostDlg.h"
#include "resource.h"
#include "RendererD3D9.h"
#include "RendererD3D11.h"
#include "RendererOGL.h"
#include "RTSSHooksInterface.h"
/////////////////////////////////////////////////////////////////////////////
// CDesktopOverlayHostWnd
/////////////////////////////////////////////////////////////////////////////
IMPLEMENT_DYNAMIC(CDesktopOverlayHostWnd, CWnd)
/////////////////////////////////////////////////////////////////////////////
#define SIZING_BORDER_SIZE			8
#define MAXIMIZE_RECT_SIZE			32
#define HOST_TIMER_ID				'HTID'
/////////////////////////////////////////////////////////////////////////////
CDesktopOverlayHostWnd::CDesktopOverlayHostWnd()
{
	m_hIcon					= NULL;
	m_hIconTray				= NULL;
	m_hCursorMaximize		= NULL;

	m_dwRenderer			= RENDERER_D3D9;
	m_lpRenderer			= NULL;

	m_dwTimerPeriod			= 0;

	m_bTransparent			= FALSE;
	m_bMaximized			= FALSE;
	m_bColorKey				= FALSE;
	m_dwBgndColor			= 0;
	m_dwAlpha				= 0;
	m_bScaleToFit			= FALSE;

	m_bCaptured				= FALSE;
	m_nCapturedX			= 0;
	m_nCapturedY			= 0;
}
/////////////////////////////////////////////////////////////////////////////
CDesktopOverlayHostWnd::~CDesktopOverlayHostWnd()
{
}
/////////////////////////////////////////////////////////////////////////////
BEGIN_MESSAGE_MAP(CDesktopOverlayHostWnd, CWnd)
	ON_WM_PAINT()
	ON_WM_SIZE()
	ON_WM_LBUTTONDOWN()
	ON_WM_LBUTTONUP()
	ON_WM_MOUSEMOVE()
	ON_WM_DESTROY()
	ON_WM_TIMER()
	ON_WM_NCHITTEST()
	ON_WM_EXITSIZEMOVE()
	ON_WM_RBUTTONDOWN()
	ON_WM_CREATE()
	ON_WM_LBUTTONDBLCLK()
	ON_WM_NCLBUTTONDOWN()
	ON_WM_SETCURSOR()
END_MESSAGE_MAP()
/////////////////////////////////////////////////////////////////////////////
// CDesktopOverlayHostWnd message handlers
/////////////////////////////////////////////////////////////////////////////
BOOL CDesktopOverlayHostWnd::InitRenderer(DWORD dwRenderer)
{
	UninitRenderer();

	m_dwRenderer =  dwRenderer;

	switch (m_dwRenderer)
	{
	case RENDERER_D3D9:
		m_lpRenderer = new CRendererD3D9;
		break;
	case RENDERER_D3D11:
		m_lpRenderer = new CRendererD3D11;
		break;
	case RENDERER_OGL:
		m_lpRenderer = new CRendererOGL;
		break;
	}

	if (m_lpRenderer)
	{
		if (m_lpRenderer->Init(m_hWnd))
			return TRUE;
	}

	if (dwRenderer != RENDERER_D3D9)
		//if we failed to init D3D11 renderer, try D3D9 fallback path
		return InitRenderer(RENDERER_D3D9);

	return FALSE;
}
/////////////////////////////////////////////////////////////////////////////
void CDesktopOverlayHostWnd::Render()
{
	if (m_lpRenderer)
		m_lpRenderer->Render(m_dwBgndColor);
}
/////////////////////////////////////////////////////////////////////////////
void CDesktopOverlayHostWnd::UninitRenderer()
{
	if (m_lpRenderer)
	{
		m_lpRenderer->Uninit();

		delete m_lpRenderer;

		m_lpRenderer = NULL;
	}
}
/////////////////////////////////////////////////////////////////////////////
void CDesktopOverlayHostWnd::OnPaint()
{
	CPaintDC dc(this); 

	if (m_lpRenderer)
		m_lpRenderer->Render(m_dwBgndColor);
}
/////////////////////////////////////////////////////////////////////////////
void CDesktopOverlayHostWnd::ResetRenderer(int nWidth, int nHeight)
{
	if (m_lpRenderer)
		m_lpRenderer->Reset(nWidth, nHeight);
}
/////////////////////////////////////////////////////////////////////////////
void CDesktopOverlayHostWnd::OnSize(UINT nType, int cx, int cy)
{
	CWnd::OnSize(nType, cx, cy);

	ResetRenderer(cx, cy);
}
/////////////////////////////////////////////////////////////////////////////
void CDesktopOverlayHostWnd::OnLButtonDown(UINT nFlags, CPoint point)
{
	if (!m_bCaptured && !m_bMaximized)
	{	
		SetCapture();

		m_bCaptured			= TRUE;
		m_nCapturedX		= point.x;
		m_nCapturedY		= point.y;

		Render();
	}

	CWnd::OnLButtonDown(nFlags, point);
}
/////////////////////////////////////////////////////////////////////////////
void CDesktopOverlayHostWnd::OnLButtonUp(UINT nFlags, CPoint point)
{
	if (m_bCaptured)
	{
		ReleaseCapture();

		m_bCaptured = FALSE;

		Render();
	}

	CWnd::OnLButtonUp(nFlags, point);
}
/////////////////////////////////////////////////////////////////////////////
void CDesktopOverlayHostWnd::OnMouseMove(UINT nFlags, CPoint point)
{
	if (m_bCaptured)
	{
		ClientToScreen(&point);

		SetWindowPos(NULL, point.x - m_nCapturedX, point.y - m_nCapturedY, 0, 0, SWP_NOSIZE);

		SaveWndPos();
	}

	CWnd::OnMouseMove(nFlags, point);
}
/////////////////////////////////////////////////////////////////////////////
void CDesktopOverlayHostWnd::InitTimer()
{
	if (m_dwTimerPeriod)
		m_mmTimer.Create(m_dwTimerPeriod, 15, 0, m_hWnd, WM_TIMER, HOST_TIMER_ID, 0);
}
/////////////////////////////////////////////////////////////////////////////
void CDesktopOverlayHostWnd::OnDestroy()
{
	m_mmTimer.Kill();
	UninitRenderer();
	DeleteTrayIcon();

	CWnd::OnDestroy();
}
/////////////////////////////////////////////////////////////////////////////
void CDesktopOverlayHostWnd::OnTimer(UINT_PTR nIDEvent)
{
	if (nIDEvent == HOST_TIMER_ID)
	{
		Render();

		m_mmTimer.EatMessages();
	}

	CWnd::OnTimer(nIDEvent);
}
/////////////////////////////////////////////////////////////////////////////
LRESULT CDesktopOverlayHostWnd::OnNcHitTest(CPoint point)
{
	LRESULT result = CWnd::OnNcHitTest(point);

	if (result == HTCLIENT)
	{
		CRect wr;
		GetWindowRect(wr);

		if (PtInRect(CRect(wr.right - MAXIMIZE_RECT_SIZE - SIZING_BORDER_SIZE, wr.top + SIZING_BORDER_SIZE, wr.right - SIZING_BORDER_SIZE, wr.top + MAXIMIZE_RECT_SIZE + SIZING_BORDER_SIZE), point))
			return HTMAXBUTTON;

		if (m_bMaximized)
			return result;

		if (point.y - wr.top < SIZING_BORDER_SIZE)
		{
			if (point.x - wr.left < SIZING_BORDER_SIZE)
				return HTTOPLEFT;
			else
			if (wr.right - point.x < SIZING_BORDER_SIZE)
				return HTTOPRIGHT;

			return HTTOP;
		}
		else
		if (wr.bottom - point.y < SIZING_BORDER_SIZE)
		{
			if (point.x - wr.left < SIZING_BORDER_SIZE)
				return HTBOTTOMLEFT;
			else
			if (wr.right - point.x < SIZING_BORDER_SIZE)
				return HTBOTTOMRIGHT;

			return HTBOTTOM;
		}
		else
		if  (point.x - wr.left < SIZING_BORDER_SIZE)
			return HTLEFT;
		else
		if (wr.right - point.x < SIZING_BORDER_SIZE)
			return HTRIGHT;

		return HTCLIENT;
	}

	return result;
}
/////////////////////////////////////////////////////////////////////////////
CString CDesktopOverlayHostWnd::GetCfgPath()
{
	char szCfgPath[MAX_PATH];
	GetModuleFileName(NULL, szCfgPath, MAX_PATH);
	PathRenameExtension(szCfgPath, ".cfg");

	return szCfgPath;
}
/////////////////////////////////////////////////////////////////////////////
CString CDesktopOverlayHostWnd::GetConfigStr(LPCSTR lpSection, LPCSTR lpName, LPCTSTR lpDefault)
{
	char szBuf[CHAR_BUF_SIZE];
	GetPrivateProfileString(lpSection, lpName, lpDefault, szBuf, CHAR_BUF_SIZE, GetCfgPath());

	return szBuf;
}
//////////////////////////////////////////////////////////////////////
void CDesktopOverlayHostWnd::SetConfigStr(LPCSTR lpSection, LPCSTR lpName, LPCSTR lpValue)
{
	WritePrivateProfileString(lpSection, lpName, lpValue, GetCfgPath());
}
//////////////////////////////////////////////////////////////////////
int	CDesktopOverlayHostWnd::GetConfigInt(LPCSTR lpSection, LPCSTR lpName, int nDefault)
{
	return GetPrivateProfileInt(lpSection, lpName, nDefault, GetCfgPath());
}
//////////////////////////////////////////////////////////////////////
void CDesktopOverlayHostWnd::SetConfigInt(LPCSTR lpSection, LPCSTR lpName, int nValue)
{
	char szValue[MAX_PATH];
	sprintf_s(szValue, sizeof(szValue), "%d", nValue);

	WritePrivateProfileString(lpSection, lpName, szValue, GetCfgPath());
}
//////////////////////////////////////////////////////////////////////
DWORD CDesktopOverlayHostWnd::GetConfigHex(LPCSTR lpSection, LPCSTR lpName, DWORD dwDefault)
{
	char szValue[MAX_PATH];
	GetPrivateProfileString(lpSection, lpName, "", szValue, MAX_PATH, GetCfgPath());

	DWORD dwResult = dwDefault; 
	sscanf_s(szValue, "%08X", &dwResult);

	return dwResult;
}
//////////////////////////////////////////////////////////////////////
void CDesktopOverlayHostWnd::SetConfigHex(LPCSTR lpSection, LPCSTR lpName, DWORD dwValue)
{
	char szValue[MAX_PATH];
	sprintf_s(szValue, sizeof(szValue), "%08X", dwValue);

	WritePrivateProfileString(lpSection, lpName, szValue, GetCfgPath());
}
//////////////////////////////////////////////////////////////////////
CRect CDesktopOverlayHostWnd::LoadWndPos()
{
	int x = GetConfigInt("Settings", "WindowX", 0);
	int y = GetConfigInt("Settings", "WindowY", 0);
	int w = GetConfigInt("Settings", "WindowW", 0);
	int h = GetConfigInt("Settings", "WindowH", 0);

	if (!w || !h)
	{
		w = 1024;
		h = 768;
	}

	return CRect(x, y, x + w, y + h);
}
//////////////////////////////////////////////////////////////////////
BOOL CDesktopOverlayHostWnd::LoadMaximizedState()
{
	m_bMaximized = GetConfigInt("Settings", "Maximized"	, 0);

	return m_bMaximized;
}
//////////////////////////////////////////////////////////////////////
void CDesktopOverlayHostWnd::SaveMaximizedState()
{
	SetConfigInt("Settings", "Maximized", m_bMaximized ? 1 : 0);
}
//////////////////////////////////////////////////////////////////////
void CDesktopOverlayHostWnd::SaveWndPos()
{
	CRect wr;

	GetWindowRect(&wr);
	
	SetConfigInt("Settings", "WindowX", wr.left);
	SetConfigInt("Settings", "WindowY", wr.top);
	SetConfigInt("Settings", "WindowW", wr.Width());
	SetConfigInt("Settings", "WindowH", wr.Height());
}
//////////////////////////////////////////////////////////////////////
void CDesktopOverlayHostWnd::OnExitSizeMove()
{
	SaveWndPos();

	CWnd::OnExitSizeMove();
}
//////////////////////////////////////////////////////////////////////
void CDesktopOverlayHostWnd::OnRButtonDown(UINT nFlags, CPoint point)
{
	CPoint cp;
	GetCursorPos(&cp);

	CMenu menu;
	menu.LoadMenu(IDR_CONTEXT_MENU);
	menu.GetSubMenu(0)->CheckMenuItem(ID_START_WITH_WINDOWS, MF_BYCOMMAND | ((m_tsi.IsTaskExist("DesktopOverlayHost") == TSI_OK) ? MF_CHECKED : MF_UNCHECKED));
	menu.GetSubMenu(0)->TrackPopupMenu(TPM_NONOTIFY, cp.x, cp.y, this);

	CWnd::OnRButtonDown(nFlags, point);
}
//////////////////////////////////////////////////////////////////////
LRESULT CDesktopOverlayHostWnd::DefWindowProc(UINT message, WPARAM wParam, LPARAM lParam)
{
	if (message == WM_TASKBARCREATED)
		RestoreTrayIcon();

	if (message == WM_COMMAND)
	{
		switch (wParam & 0xFFFF)
		{
		case ID_SETUP:
			OnSetup();
			break;
		case ID_START_WITH_WINDOWS:
			OnStartWithWindows();
			break;
		case ID_CLOSE:
			PostMessage(WM_QUIT);
			break;
		}
	}

	if (message == WM_TRAYNOTIFICATION)
	{
		switch (lParam) 
		{
		case WM_LBUTTONUP:
			if (GetAsyncKeyState(VK_CONTROL) < 0)
			{
				if (m_bMaximized)
				{
					m_bMaximized = FALSE;

					ShowWindow(SW_RESTORE);
				}

				SetWindowPos(NULL, 0, 0, 0, 0, SWP_NOSIZE);

				SaveMaximizedState();
				SaveWndPos();
			}
			else
				OnSetup();
			break;
		case WM_RBUTTONUP:
			CPoint cp;
			GetCursorPos(&cp);

			SetForegroundWindow();

			CMenu menu;
			menu.LoadMenu(IDR_CONTEXT_MENU);
			menu.GetSubMenu(0)->TrackPopupMenu(TPM_NONOTIFY, cp.x, cp.y, this);
			break;
		}
	}



	return CWnd::DefWindowProc(message, wParam, lParam);
}
//////////////////////////////////////////////////////////////////////
int CDesktopOverlayHostWnd::OnCreate(LPCREATESTRUCT lpCreateStruct)
{
	if (CWnd::OnCreate(lpCreateStruct) == -1)
		return -1;

	m_hIcon					= (HICON)LoadImage(AfxGetInstanceHandle(),MAKEINTRESOURCE(IDR_MAINFRAME),IMAGE_ICON,32,32,LR_DEFAULTCOLOR);
	m_hIconTray				= (HICON)LoadImage(AfxGetInstanceHandle(),MAKEINTRESOURCE(IDR_MAINFRAME),IMAGE_ICON,16,16,LR_DEFAULTCOLOR);

	m_hCursorMaximize		= LoadCursor(NULL, IDC_SIZEALL);

	SetIcon(m_hIcon		, TRUE);
	SetIcon(m_hIconTray	, FALSE);

	ZeroMemory(&m_nidDesc, sizeof(m_nidDesc));

	m_nidDesc.nid.cbSize			= sizeof(m_nidDesc.nid); 
	m_nidDesc.nid.hWnd				= m_hWnd;
	m_nidDesc.nid.uID				= (UINT)m_hWnd;
	m_nidDesc.nid.uFlags			= NIF_ICON | NIF_TIP | NIF_MESSAGE;
	m_nidDesc.nid.uCallbackMessage	= WM_TRAYNOTIFICATION;
	m_nidDesc.nid.hIcon				= m_hIconTray;

	strcpy_s(m_nidDesc.nid.szTip, sizeof(m_nidDesc.nid.szTip), "Desktop Overlay Host v1.1.0");

	UpdateAttributes();

	AddTrayIcon();

	m_dpi.SetDpiAware(DPI_AWARE_PER_MONITOR_V2);
	m_tsi.Init();

	return 0;
}
//////////////////////////////////////////////////////////////////////
void CDesktopOverlayHostWnd::AddTrayIcon()
{
	if (!m_nidDesc.flags)
	{
		MyShell_NotifyIcon(NIM_ADD,(NOTIFYICONDATA*)&m_nidDesc.nid);

		m_nidDesc.flags = 1;
	}
}
/////////////////////////////////////////////////////////////////////////////
void CDesktopOverlayHostWnd::DeleteTrayIcon()
{
	if (m_nidDesc.flags)
	{
		Shell_NotifyIcon(NIM_DELETE,(NOTIFYICONDATA*)&m_nidDesc.nid);

		m_nidDesc.flags = 0;
	}
}
/////////////////////////////////////////////////////////////////////////////
BOOL CDesktopOverlayHostWnd::MyShell_NotifyIcon(DWORD dwMessage, PNOTIFYICONDATA lpdata)
{
	for (DWORD dwRetry=0; dwRetry<5; dwRetry++)
	{
		if (Shell_NotifyIcon(dwMessage, lpdata))
			return TRUE;

		Sleep(1000);
	}

	return FALSE;
}
/////////////////////////////////////////////////////////////////////////////
void CDesktopOverlayHostWnd::RestoreTrayIcon()
{
	if (m_nidDesc.flags)
		MyShell_NotifyIcon(NIM_ADD,(NOTIFYICONDATA*)&m_nidDesc.nid);
}
/////////////////////////////////////////////////////////////////////////////
void CDesktopOverlayHostWnd::LoadSettings()
{
	m_dwTimerPeriod	= GetConfigInt("Settings", "TimerPeriod", 33);

	m_bTransparent	= GetConfigInt("Settings", "Transparent", 0);
	m_bColorKey		= GetConfigInt("Settings", "ColorKey"	, 0);
	m_dwBgndColor	= GetConfigHex("Settings", "BgndColor"	, 0x800080);
	m_dwAlpha		= GetConfigHex("Settings", "Alpha"		, 0xFF);
	m_bScaleToFit	= GetConfigInt("Settings", "ScaleToFit"	, 0);
}
/////////////////////////////////////////////////////////////////////////////
void CDesktopOverlayHostWnd::SaveSettings()
{
	SetConfigInt("Settings", "Transparent"	, m_bTransparent	? 1 : 0);
	SetConfigInt("Settings", "ColorKey"		, m_bColorKey		? 1 : 0);
	SetConfigHex("Settings", "BgndColor"	, m_dwBgndColor);
	SetConfigHex("Settings", "Alpha"		, m_dwAlpha);
	SetConfigInt("Settings", "Renderer"		, m_dwRenderer);
	SetConfigInt("Settings", "ScaleToFit"	, m_bScaleToFit);
}
/////////////////////////////////////////////////////////////////////////////
void CDesktopOverlayHostWnd::OnSetup()
{
	CDesktopOverlayHostDlg dlg;

	dlg.m_bTransparent			= m_bTransparent;
	dlg.m_bColorKey				= m_bColorKey;
	dlg.m_dwBgndColor			= m_dwBgndColor;
	dlg.m_dwAlpha				= m_dwAlpha;
	dlg.m_nRenderer				= m_dwRenderer;
	dlg.m_bScaleToFit			= m_bScaleToFit;

	if (IDOK == dlg.DoModal())
	{
		m_bTransparent		= dlg.m_bTransparent;
		m_bColorKey			= dlg.m_bColorKey;
		m_dwBgndColor		= dlg.m_dwBgndColor;
		m_dwAlpha			= dlg.m_dwAlpha;

		UpdateAttributes();

		if (m_dwRenderer != dlg.m_nRenderer)
			InitRenderer(dlg.m_nRenderer);

		if (m_bScaleToFit != dlg.m_bScaleToFit)
		{
			m_bScaleToFit = dlg.m_bScaleToFit;

			UpdateScaleToFit();
		}

		SaveSettings();
	}
}
/////////////////////////////////////////////////////////////////////////////
void CDesktopOverlayHostWnd::OnStartWithWindows()
{
	if (m_tsi.IsTaskExist("DesktopOverlayHost") == TSI_OK)
	{
		int err = m_tsi.DeleteTask("DesktopOverlayHost");

		if (err)
			MessageBox("Failed to delete startup task!", "DesktopOverlayHost", MB_ICONWARNING|MB_OK);
	}
	else
	{
		char szPath[MAX_PATH];
		GetModuleFileName(NULL, szPath, MAX_PATH);

		int err = m_tsi.AddTask("DesktopOverlayHost", "", szPath, "");

		if (err)
			MessageBox("Failed to add startup task!", "DesktopOverlayHost", MB_ICONWARNING|MB_OK);
	}
}
/////////////////////////////////////////////////////////////////////////////
void CDesktopOverlayHostWnd::UpdateAttributes()
{
	SetLayeredWindowAttributes(m_dwBgndColor, (BYTE)m_dwAlpha, LWA_ALPHA | (m_bColorKey ? LWA_COLORKEY : 0)); 

	if (m_bTransparent) 
		ModifyStyleEx(0, WS_EX_TRANSPARENT);
	else
		ModifyStyleEx(WS_EX_TRANSPARENT, 0);
}
/////////////////////////////////////////////////////////////////////////////
void CDesktopOverlayHostWnd::OnLButtonDblClk(UINT nFlags, CPoint point)
{
	OnSetup();

	CWnd::OnLButtonDblClk(nFlags, point);
}
/////////////////////////////////////////////////////////////////////////////
BOOL CDesktopOverlayHostWnd::IsRTSSInjected()
{
	return GetModuleHandle("RTSSHooks.dll") != NULL;
}
/////////////////////////////////////////////////////////////////////////////
void CDesktopOverlayHostWnd::UpdateScaleToFit()
{
	DWORD dwScaleToFit = m_bScaleToFit ? 1 : 0;

	CRTSSHooksInterface rtssInterface;
/*
	rtssInterface.LoadProfile("DesktopOverlayHost.exe");

	if (rtssInterface.SetProfileProperty("ScaleToFit", (LPBYTE)&dwScaleToFit, sizeof(dwScaleToFit)))
	{
		rtssInterface.SaveProfile("DesktopOverlayHost.exe");
		rtssInterface.UpdateProfiles();
	}
*/

	//NOTE: we could modify ScaleToFit profile property in file system via profile API, but instead of doing so we'll
	//use more efficient way and modify ScaleToFit for the profile loaded in memory with special profile override environment variable

	CString strProfileOverride;
	strProfileOverride.Format("ScaleToFit,%d", dwScaleToFit);

	SetEnvironmentVariable("RTSSHooksProfileOverride", strProfileOverride);

	rtssInterface.UpdateProfiles();
}
/////////////////////////////////////////////////////////////////////////////
BOOL CDesktopOverlayHostWnd::GetScaleToFit()
{
	CRTSSHooksInterface rtssInterface;

	rtssInterface.LoadProfile("DesktopOverlayHost.exe");

	DWORD dwProperty;

	if (rtssInterface.GetProfileProperty("ScaleToFit", (LPBYTE)&dwProperty, sizeof(dwProperty)))
		return dwProperty;

	return FALSE;
}
/////////////////////////////////////////////////////////////////////////////
void CDesktopOverlayHostWnd::SetScaleToFit(BOOL bScaleToFit)
{
	CRTSSHooksInterface rtssInterface;

	rtssInterface.LoadProfile("DesktopOverlayHost.exe");

	DWORD dwProperty = bScaleToFit ? 1 : 0;

	if (rtssInterface.SetProfileProperty("ScaleToFit", (LPBYTE)&dwProperty, sizeof(dwProperty)))
	{
		rtssInterface.SaveProfile("DesktopOverlayHost.exe");
		rtssInterface.UpdateProfiles();
	}
}
/////////////////////////////////////////////////////////////////////////////
void CDesktopOverlayHostWnd::OnNcLButtonDown(UINT nHitTest, CPoint point)
{
	if (nHitTest == HTMAXBUTTON)
	{
		m_bMaximized = !m_bMaximized;

		SaveMaximizedState();

		if (m_bMaximized) 
			ShowWindow(SW_MAXIMIZE);
		else
			ShowWindow(SW_RESTORE);
	}

	CWnd::OnNcLButtonDown(nHitTest, point);
}
/////////////////////////////////////////////////////////////////////////////
BOOL CDesktopOverlayHostWnd::OnSetCursor(CWnd* pWnd, UINT nHitTest, UINT message)
{
	if (nHitTest == HTMAXBUTTON)
	{
		::SetCursor(m_hCursorMaximize);

		return TRUE;
	}

	return CWnd::OnSetCursor(pWnd, nHitTest, message);
}
/////////////////////////////////////////////////////////////////////////////
